<?php
/**
 * @package     Joomla.Administrator
 * @subpackage  com_linguise
 *
 * @copyright   Copyright (C) 2005 - 2020 Open Source Matters, Inc. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

defined('_JEXEC') or die;

/**
 * Linguise component helper.
 */
class LinguiseHelper extends JHelperContent
{
    /**
     * Store options in this array
     * @var array
     */
    static protected $_options = [
        'linguise_field_token' => '',
        'moduleclass_sfx' => '',
        'language_default' => 'en',
        'languages_enabled' => array(),
        'flag_display_type' => 'popup',
        'display_position' => 'no',
        'enable_flag' => 1,
        'enable_language_name' => 1,
        'flag_shape' => 'rounded',
        'flag_border_radius' => 0,
        'flag_width' => 24,
        'browser_redirect' => 0,
        'language_name_display' => 'en',
        'pre_text' => '',
        'post_text' => '',
        'alternate_link' => 1,
        'custom_css' => '',
        'cache_enabled' => 1,
        'cache_max_size' => 200,
        'language_name_color' => '#222',
        'language_name_hover_color' => '#222',
        'flag_shadow_h' => 3,
        'flag_shadow_v' => 3,
        'flag_shadow_blur' => 6,
        'flag_shadow_spread' => 0,
        'flag_shadow_color' => '#bfbfbf',
        'flag_hover_shadow_h' => 3,
        'flag_hover_shadow_v' => 3,
        'flag_hover_shadow_blur' => 6,
        'flag_hover_shadow_spread' => 0,
        'flag_hover_shadow_color' => '#bfbfbf',


        // Config variables created on the fly, not saved into db
        'languages' => [],
        'base' => null,
        'original_path' => null,
        'trailing_slashes' => null,
        'joomla_languages' => [],
        'joomla_default_language' => null,
        'current_language' => null,
    ];

    /**
     * List of public options, that can be shared in frontend
     * @var array
     */
    static protected $_public_options = [
        'flag_display_type',
        'display_position',
        'enable_flag',
        'enable_language_name',
        'flag_shape',
        'pre_text',
        'post_text',
        'joomla_languages',
        'current_language',
        'language_default',
        'languages',
        'base',
        'original_path',
        'trailing_slashes',
    ];

    static $_options_initialized = false;

    /**
     * Retrieve Linguise option
     *
     * @param string
     * @param mixed
     * @return mixed|null
     */
    public static function getOption($name, $default = null)
    {
        if (!self::$_options_initialized) {
            self::getOptions();
        }

        if (!isset(self::$_options[$name])) {
            return $default;
        }

        return self::$_options[$name];
    }

	/**
	 * Returns linguise configs
	 *
	 * @return  array
	 */
	public static function getOptions($reset = false)
	{

        if (self::$_options_initialized && $reset === false) {
            return self::$_options;
        }

        self::$_options_initialized = true;

        $saved_options = LinguiseHelper::getLinguiseOptions();
        if (!empty($saved_options)) {
            $options_value = json_decode($saved_options['params'], true);
            if (!empty($options_value)) {
                self::$_options = array_merge(self::$_options, $options_value);
            }
        }

        if (JLanguageMultilang::isEnabled()) {
            $factory_language = JFactory::getLanguage();
            $joomla_languages = \Joomla\CMS\Language\LanguageHelper::getInstalledLanguages(0);

            self::$_options['joomla_default_language'] = substr($factory_language->getDefault(), 0, 2);

            foreach ($joomla_languages as $joomla_language) {
                if ($joomla_language->client_id !== "0") {
                    continue;
                }
                $language = substr($joomla_language->element, 0, 2);

                self::$_options['joomla_languages'][] = $language;
            }

        }

		return self::$_options;
	}

    /**
     * Update the value of an option in the helper
     *
     * @param $name
     * @param $value
     * @return void
     */
    public static function setOption($name, $value) {
        if (!self::$_options_initialized) {
            self::getOptions();
        }

        self::$_options[$name] = $value;
    }

    /**
     * Return the options that are safe to display in frontend
     * @return array
     */
    public static function getPublicOptions()
    {
        if (!self::$_options_initialized) {
            self::getOptions();
        }

        return array_intersect_key(self::$_options, array_flip(self::$_public_options));
    }

    /**
     * Sanitizes a string key.
     *
     * Keys are used as internal identifiers. Lowercase alphanumeric characters,
     * dashes, and underscores are allowed.
     *
     * @param string $key String key
     *
     * @return string Sanitized key
     */
    public static function sanitize_key($key)
    {
        $raw_key = $key;
        $key = strtolower($key);
        $key = preg_replace('/[^a-z0-9_\-]/', '', $key);
        return $key;
    }

    /**
     * Outputs the HTML checked attribute.
     *
     * @param mixed   $checked One of the values to compare
     * @param mixed   $current The other value to compare if not just true
     * @param boolean $echo    Whether to echo or just return the string
     *
     * @return string HTML attribute or empty string
     */
    public static function checked($checked, $current = true, $echo = true)
    {
        return LinguiseHelper::__checked_selected_helper($checked, $current, $echo, 'checked');
    }

    /**
     * Private helper function for checked, selected, disabled and readonly.
     *
     * Compares the first two arguments and if identical marks as $type
     *
     * @param mixed   $helper  One of the values to compare
     * @param mixed   $current (true) The other value to compare if not just true
     * @param boolean $echo    Whether to echo or just return the string
     * @param string  $type    The type of checked|selected|disabled|readonly we are doing
     *
     * @return string HTML attribute or empty string
     */
    public static function __checked_selected_helper($helper, $current, $echo, $type)
    { // phpcs:ignore WordPress.NamingConventions.ValidFunctionName.FunctionDoubleUnderscore,PHPCompatibility.FunctionNameRestrictions.ReservedFunctionNames.FunctionDoubleUnderscore
        if ((string)$helper === (string)$current) {
            $result = " $type='$type'";
        } else {
            $result = '';
        }

        if ($echo) {
            echo $result;
        }

        return $result;
    }

    /**
     * Outputs the HTML selected attribute.
     *
     * Compares the first two arguments and if identical marks as selected
     *
     * @param mixed   $selected One of the values to compare
     * @param mixed   $current  (true) The other value to compare if not just true
     * @param boolean $echo     Whether to echo or just return the string
     *
     * @return string HTML attribute or empty string
     */
    public static function selected($selected, $current = true, $echo = true)
    {
        return LinguiseHelper::__checked_selected_helper($selected, $current, $echo, 'selected');
    }

    /**
     * Get linguise Options
     *
     * @return boolean|mixed
     */
    public static function getLinguiseOptions()
    {
        $db = JFactory::getDbo();
        $query = 'SELECT * FROM #__extensions WHERE element = "linguise"';
        $db->setQuery($query);
        if (!$db->execute()) {
            return false;
        }
        return $db->loadAssoc();
    }

    /**
     * Get linguise module Options
     *
     * @return boolean|mixed
     */
    public static function getLinguiseModuleOptions()
    {
        $db = JFactory::getDbo();
        $query = 'SELECT * FROM #__modules WHERE module = "mod_linguise"';
        $db->setQuery($query);
        if (!$db->execute()) {
            return false;
        }
        return $db->loadAssoc();
    }

    /**
     * Get linguise plugin Options
     *
     * @return boolean|mixed
     */
    public static function getLinguisePluginOptions()
    {
        $db = JFactory::getDbo();
        $query = 'SELECT * FROM #__extensions WHERE element = "linguise"';
        $db->setQuery($query);
        if (!$db->execute()) {
            return false;
        }
        return $db->loadAssoc();
    }

    /**
     * Insert option
     *
     * @param array $options
     *
     * @return void
     */
    public static function insertLinguiseOptions($options = array())
    {
        $db = JFactory::getDbo();
        $query = $db->getQuery(true);

        $columns = array('element', 'params');
        $values = array($db->quote('linguise'), $db->quote(json_encode($options)));

        $query
            ->insert($db->quoteName('#__extensions'))
            ->columns($db->quoteName($columns))
            ->values(implode(',', $values));

        $db->setQuery($query);
        $db->execute();
    }

    /**
     * Update option
     *
     * @param array $options
     *
     * @return void
     */
    public static function updateLinguiseOptions($options = array())
    {
        $db = JFactory::getDbo();
        $query = $db->getQuery(true);
        $fields = array(
            $db->quoteName('params') . ' = ' . $db->quote(json_encode($options)),
        );

        $conditions = array(
            $db->quoteName('element') . ' = "linguise"',
        );

        $query->update($db->quoteName('#__extensions'))->set($fields)->where($conditions);
        $db->setQuery($query);
        $db->execute();
    }

    /**
     * Render custom CSS
     *
     * @param array  $options    Options
     * @param string $custom_css Custom CSS string
     *
     * @return string
     */
    public static function linguiseRenderCustomCss($options, $custom_css = '')
    {
        if ($options['flag_shape'] === 'rounded') {
            $custom_css .= '
                .linguise_switcher span.linguise_language_icon, #linguise_popup li span.linguise_flags {
                        width: ' . (int)$options['flag_width'] . 'px !important;
                        height: ' . (int)$options['flag_width'] . 'px !important;
                }';
        } else {
            $custom_css .= '
                .linguise_switcher span.linguise_language_icon, #linguise_popup li span.linguise_flags {
                        width: ' . (int)$options['flag_width'] . 'px !important;
                        height: ' . ((int)$options['flag_width']*2/3) . 'px !important;
                }';
        }
        $custom_css .= '.linguise_lang_name {color: '. htmlspecialchars($options['language_name_color']) .' !important}';
        $custom_css .= '.linguise_lang_name:hover {color: '. htmlspecialchars($options['language_name_hover_color']) .' !important}';
        $custom_css .= '.linguise_switcher span.linguise_language_icon {box-shadow: '. (int)$options['flag_shadow_h'] .'px '. (int)$options['flag_shadow_v'] .'px '. (int)$options['flag_shadow_blur'] .'px '. (int)$options['flag_shadow_spread'] .'px '. htmlspecialchars($options['flag_shadow_color']) .' !important}';
        $custom_css .= '.linguise_switcher span.linguise_language_icon:hover {box-shadow: '. (int)$options['flag_hover_shadow_h'] .'px '. (int)$options['flag_hover_shadow_v'] .'px '. (int)$options['flag_hover_shadow_blur'] .'px '. (int)$options['flag_hover_shadow_spread'] .'px '. htmlspecialchars($options['flag_hover_shadow_color']) .' !important}';
        if ($options['flag_shape'] === 'rectangular') {
            $custom_css .= '#linguise_popup.linguise_flag_rectangular ul li .linguise_flags, .linguise_switcher.linguise_flag_rectangular span.linguise_language_icon {border-radius: '. (int)$options['flag_border_radius'] .'px !important}';
        }
        if (!empty($options['custom_css'])) {
            $custom_css .= htmlspecialchars($options['custom_css']);
        }

        return $custom_css;
    }

    /**
     * Linguise Force Relative Url
     *
     * @param string $url Url
     *
     * @return null|string|string[]
     */
    public static function linguiseForceRelativeUrl($url)
    {
        return preg_replace('/^(http)?s?:?\/\/[^\/]*(\/?.*)$/i', '$2', '' . $url);
    }

    /**
     * Redirect
     *
     * @param array $options Linguise options
     *
     * @return void
     */
    public static function browserRedirect()
    {
        $options = self::getOptions();

        // Check redirect option enabled
        if (empty($options['browser_redirect'])) {
            return;
        }

        // We don't redirect visitors that are already in translated page
        if ($options['current_language'] !== $options['language_default']) {
            return;
        }

        // Only redirect GET request
        if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
            return;
        }

        // Check that a language has been set
        if (empty($_SERVER['HTTP_ACCEPT_LANGUAGE']) && empty($_SERVER['HTTP_CF_IPCOUNTRY'])) { //phpcs:ignore
            return;
        }

        // Check we don't have already made a redirect for this visitor
        if (!empty($_COOKIE['LINGUISE_REDIRECT'])) {
            return;
        }

        // Do not redirect from internal visit
        $uri = JUri::getInstance();
        if (!empty($_SERVER['HTTP_REFERER']) || strpos($_SERVER['HTTP_REFERER'], $uri::root()) === 0) {
            return;
        }

        // Avoid redirecting internal crawlers
        if (!empty($_SERVER['HTTP_USER_AGENT']) && $_SERVER['HTTP_USER_AGENT'] === 'RSSeo! Crawler') {
            return;
        }

        if (isset($_SERVER['HTTP_ACCEPT_LANGUAGE'])) { //phpcs:ignore
            $browser_language = substr($_SERVER['HTTP_ACCEPT_LANGUAGE'], 0, 2); // fixme: won't work for zh-cn
        } elseif (isset($_SERVER['HTTP_CF_IPCOUNTRY'])) { // phpcs:ignore
            // Cloudfare Compatibility
            $browser_language = strtolower($_SERVER['HTTP_CF_IPCOUNTRY']); //phpcs:ignore
        } else {
            // No browser language found
            return;
        }

        if (!$browser_language) {
            return;
        }

        if (isset($_SERVER['HTTP_LINGUISE_ORIGINAL_LANGUAGE'])) {
            $url_language = $_SERVER['HTTP_LINGUISE_ORIGINAL_LANGUAGE'];
        } else {
            $url_language = isset($options['language_default']) ? $options['language_default'] : 'en';
        }

        if ($url_language === $browser_language) {
            return;
        }

        if (!in_array($browser_language, $options['languages_enabled'])) {
            return;
        }

        $base = rtrim($uri->base(true),'/');

        $protocol = 'http';
        if (strpos($uri::root(), 'https') === 0) {
            $protocol = 'https';
        }

        if (in_array($browser_language, $options['joomla_languages'])) {
            // This is a language handled by Joomla
            JLoader::register('ModLanguagesHelper', JPATH_ROOT . '/modules/mod_languages/helper.php');
            $joomla_language_list = ModLanguagesHelper::getList($params);
            foreach ($joomla_language_list as $joomla_language) {
                $jlang = substr($joomla_language->lang_code, 0, 2);
                if ($jlang !== $browser_language) {
                    continue;
                }
                $url_auto_redirect = $protocol . '://' . $_SERVER['HTTP_HOST'] . $joomla_language->link;
                break;
            }
        }

        if (empty($url_auto_redirect)) {
            $path = rtrim(substr(rtrim(parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH), '/'), strlen($base)), '/');

            // Set the default value to the root of the website in translated language
            $url_auto_redirect = $protocol . '://' . $_SERVER['HTTP_HOST'] . $base . '/' . $browser_language . $path;
        }

        if (!empty($_SERVER['QUERY_STRING'])) {
            $url_auto_redirect .= '/?' . $_SERVER['QUERY_STRING'];
        }
        setcookie('LINGUISE_REDIRECT', 1, time() + 20);
        header('Linguise-Translated-Redirect: 1');
        header('Cache-Control: no-cache, must-revalidate, max-age=0');
        header('Location: ' . $url_auto_redirect, true, 302);
        exit();
    }

    public static function addAssets() {
        $document = JFactory::getDocument();
        $document->addScript(JUri::root() . 'modules/mod_linguise/assets/js/front.bundle.js');
        $document->addStyleSheet(JUri::root() . 'modules/mod_linguise/assets/css/front.bundle.css');

        if (empty($document->_script['linguise/json'])) {
            $document->addScriptDeclaration(json_encode(LinguiseHelper::getPublicOptions()), 'linguise/json');
        }

    }
}
